from flask_sqlalchemy import SQLAlchemy
from sqlalchemy import CheckConstraint
import enum

db = SQLAlchemy()


class CarType(enum.Enum):
    suv = 'suv'
    small = 'small'
    van = 'van'


class Car(db.Model):
    __tablename__ = 'cars'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(128), nullable=False)
    seats = db.Column(db.Integer, nullable=False)
    ports = db.Column(db.Integer, nullable=False)
    type = db.Column(db.Enum(CarType), nullable=False)
    brand = db.Column(db.String(64), nullable=False)
    license_plate = db.Column(db.String(64), nullable=False, unique=True)

    reservations = db.relationship('Reservation', backref='car', cascade='all, delete-orphan')

    __table_args__ = (
        CheckConstraint('ports IN (3,5)', name='check_ports'),
    )

    def to_dict(self):
        return {
            'id': self.id,
            'name': self.name,
            'seats': self.seats,
            'ports': self.ports,
            'type': self.type.value if self.type else None,
            'brand': self.brand,
            'license_plate': self.license_plate,
        }


class Reservation(db.Model):
    __tablename__ = 'reservations'

    id = db.Column(db.Integer, primary_key=True)
    pickup_date = db.Column(db.DateTime, nullable=False)
    return_date = db.Column(db.DateTime, nullable=False)
    car_id = db.Column(db.Integer, db.ForeignKey('cars.id'), nullable=False)
    customer_fullname = db.Column(db.String(128), nullable=False)
    drive_license = db.Column(db.String(64), nullable=False)

    def to_dict(self):
        return {
            'id': self.id,
            'pickup_date': self.pickup_date.isoformat() if self.pickup_date else None,
            'return_date': self.return_date.isoformat() if self.return_date else None,
            'car_id': self.car_id,
            'customer_fullname': self.customer_fullname,
            'drive_license': self.drive_license,
        }
