import json
from datetime import datetime, timedelta

import pytest

from app import create_app
from models import db, Car


@pytest.fixture
def app():
    app = create_app()
    app.config.update({
        'TESTING': True,
        # Use in-memory SQLite for tests
        'SQLALCHEMY_DATABASE_URI': 'sqlite:///:memory:',
    })

    with app.app_context():
        db.create_all()
        yield app
        db.session.remove()
        db.drop_all()


@pytest.fixture
def client(app):
    return app.test_client()


def test_create_and_get_car(client):
    payload = {
        'name': 'Test Car',
        'seats': 4,
        'ports': 3,
        'type': 'small',
        'brand': 'TestBrand',
        'license_plate': 'TEST-123'
    }
    rv = client.post('/cars', json=payload)
    assert rv.status_code == 201
    data = rv.get_json()
    assert data['name'] == 'Test Car'

    # GET list
    rv2 = client.get('/cars')
    assert rv2.status_code == 200
    arr = rv2.get_json()
    assert any(c['license_plate'] == 'TEST-123' for c in arr)


def test_create_reservation(client):
    # Create car first
    car_payload = {
        'name': 'Reserve Car',
        'seats': 5,
        'ports': 5,
        'type': 'suv',
        'brand': 'BrandX',
        'license_plate': 'RES-001'
    }
    rv = client.post('/cars', json=car_payload)
    assert rv.status_code == 201
    car = rv.get_json()

    pickup = datetime.utcnow()
    return_dt = pickup + timedelta(days=2)

    res_payload = {
        'pickup_date': pickup.isoformat(),
        'return_date': return_dt.isoformat(),
        'car_id': car['id'],
        'customer_fullname': 'John Tester',
        'drive_license': 'DL-0001'
    }

    rv2 = client.post('/reservations', json=res_payload)
    assert rv2.status_code == 201
    res = rv2.get_json()
    assert res['car_id'] == car['id']